/**
 * Zdravo AI Extension Popup
 * Handles authentication, conversation capture, and memory saving
 */

// Configuration
const CONFIG = {
  API_URL: 'https://www.zdravoai.com',
  MAX_RETRIES: 3,
  RETRY_DELAY: 1000,
};

// State management
let isCapturing = false;

/**
 * Check authentication status - uses background script's stored token
 */
async function checkAuth() {
  try {
    // Get token from background script
    const response = await chrome.runtime.sendMessage({ action: 'getAuthToken' });
    const authToken = response?.token;
    
    if (authToken) {
      // Decode JWT to get user info (basic validation)
      try {
        const payload = JSON.parse(atob(authToken.split('.')[1]));
        const userEmail = payload.email || payload.sub;
        const userId = payload.sub;
        
        // Check if token is expired
        const now = Date.now() / 1000;
        if (payload.exp && payload.exp < now) {
          // Token expired
          await chrome.runtime.sendMessage({ action: 'clearAuthToken' });
          return { isAuthenticated: false, userEmail: null, userId: null, token: null };
        }
        
        return { 
          isAuthenticated: true, 
          userEmail: userEmail || 'User', 
          userId: userId,
          token: authToken
        };
      } catch (decodeError) {
        console.error('[Zdravo AI] Token decode error:', decodeError);
        return { isAuthenticated: false, userEmail: null, userId: null, token: null };
      }
    }
    
    return { isAuthenticated: false, userEmail: null, userId: null, token: null };
  } catch (error) {
    console.error('[Zdravo AI] Auth check error:', error);
    return { isAuthenticated: false, userEmail: null, userId: null, token: null };
  }
}

/**
 * Get current active tab
 */
async function getCurrentTab() {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
  return tab;
}

/**
 * Detect AI platform from URL
 */
function detectPlatform(url) {
  if (!url) return null;
  if (url.includes('chat.openai.com') || url.includes('chatgpt.com')) return 'chatgpt';
  if (url.includes('claude.ai')) return 'claude';
  if (url.includes('gemini.google.com')) return 'gemini';
  if (url.includes('perplexity.ai')) return 'perplexity';
  return null;
}

/**
 * Get platform display name
 */
function getPlatformDisplayName(platform) {
  const names = {
    chatgpt: 'ChatGPT',
    claude: 'Claude',
    gemini: 'Gemini',
    perplexity: 'Perplexity',
  };
  return names[platform] || platform;
}

/**
 * Save memory to backend with retry logic
 */
async function saveMemory(memory, authToken, retryCount = 0) {
  try {
    const headers = {
      'Content-Type': 'application/json',
    };
    
    if (authToken) {
      headers['Authorization'] = `Bearer ${authToken}`;
    }
    
    const response = await fetch(`${CONFIG.API_URL}/api/memories`, {
      method: 'POST',
      headers: headers,
      body: JSON.stringify(memory),
    });

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({}));

      // Handle specific error cases
      if (response.status === 403 && errorData.limit) {
        throw new Error(`LIMIT_REACHED:${errorData.limit}`);
      }

      if (response.status === 401) {
        // Clear expired token
        await chrome.runtime.sendMessage({ action: 'clearAuthToken' });
        throw new Error('UNAUTHORIZED');
      }

      throw new Error(errorData.error || `HTTP ${response.status}`);
    }

    return await response.json();
  } catch (error) {
    // Retry logic for network errors
    if (retryCount < CONFIG.MAX_RETRIES && error.message !== 'LIMIT_REACHED' && error.message !== 'UNAUTHORIZED') {
      await new Promise(resolve => setTimeout(resolve, CONFIG.RETRY_DELAY * (retryCount + 1)));
      return saveMemory(memory, authToken, retryCount + 1);
    }
    throw error;
  }
}

/**
 * Inject content script if not already present
 */
async function ensureContentScriptInjected(tabId, platform) {
  try {
    // Try to ping the content script
    await chrome.tabs.sendMessage(tabId, { action: 'ping' });
    return true;
  } catch {
    // Content script not loaded, try to inject it
    const scriptMap = {
      chatgpt: 'content-scripts/chatgpt.js',
      claude: 'content-scripts/claude.js',
      gemini: 'content-scripts/gemini.js',
      perplexity: 'content-scripts/perplexity.js',
    };
    
    const scriptFile = scriptMap[platform];
    if (!scriptFile) return false;
    
    try {
      await chrome.scripting.executeScript({
        target: { tabId },
        files: [scriptFile],
      });
      // Wait a moment for script to initialize
      await new Promise(resolve => setTimeout(resolve, 500));
      return true;
    } catch (injectError) {
      console.error('[Zdravo AI] Failed to inject content script:', injectError);
      return false;
    }
  }
}

/**
 * Capture conversation from current tab
 */
async function captureConversation() {
  if (isCapturing) return;
  
  isCapturing = true;
  const button = document.getElementById('captureBtn');
  const originalText = button.textContent;
  button.disabled = true;
  button.textContent = 'Capturing...';
  
  try {
    const tab = await getCurrentTab();
    const platform = detectPlatform(tab.url);
    
    if (!platform) {
      throw new Error('UNSUPPORTED_PLATFORM');
    }
    
    // Ensure content script is injected
    const isInjected = await ensureContentScriptInjected(tab.id, platform);
    if (!isInjected) {
      throw new Error('INJECTION_FAILED');
    }
    
    // Send message to content script to extract conversation
    const response = await chrome.tabs.sendMessage(tab.id, {
      action: 'extractConversation',
    });

    if (!response || !response.success) {
      throw new Error(response?.error || 'EXTRACTION_FAILED');
    }
    
    // Save the memory
    const memory = {
      title: response.title || `Untitled ${getPlatformDisplayName(platform)} Conversation`,
      content: response.content,
      sourcePlatform: platform,
      url: tab.url,
      tags: response.tags || [platform],
    };
    
    const result = await saveMemory(memory);
    showSuccess(result.memory);
    
  } catch (error) {
    console.error('[Zdravo AI] Capture error:', error);
    handleCaptureError(error.message);
  } finally {
    isCapturing = false;
    button.disabled = false;
    button.textContent = originalText;
  }
}

/**
 * Handle capture errors with user-friendly messages
 */
function handleCaptureError(errorCode) {
  const errorMessages = {
    'UNSUPPORTED_PLATFORM': {
      title: 'Unsupported Platform',
      message: 'Please navigate to ChatGPT, Claude, Gemini, or Perplexity to capture conversations.',
      action: 'Go to ChatGPT',
      actionUrl: 'https://chat.openai.com',
    },
    'INJECTION_FAILED': {
      title: 'Page Not Ready',
      message: 'The page is still loading. Please wait a moment and try again.',
      action: null,
      actionUrl: null,
    },
    'EXTRACTION_FAILED': {
      title: 'Extraction Failed',
      message: 'Could not find conversation on this page. Make sure a conversation is loaded.',
      action: null,
      actionUrl: null,
    },
    'LIMIT_REACHED': {
      title: 'Memory Limit Reached',
      message: 'You\'ve reached your 50 memory limit for this month. Upgrade to Pro for unlimited memories.',
      action: 'Upgrade Now',
      actionUrl: `${CONFIG.API_URL}/pricing`,
    },
    'UNAUTHORIZED': {
      title: 'Session Expired',
      message: 'Please sign in again to continue saving memories.',
      action: 'Sign In',
      actionUrl: `${CONFIG.API_URL}/auth/login`,
    },
    'NETWORK_ERROR': {
      title: 'Connection Error',
      message: 'Please check your internet connection and try again.',
      action: null,
      actionUrl: null,
    },
  };
  
  const error = errorMessages[errorCode] || {
    title: 'Capture Failed',
    message: 'An unexpected error occurred. Please try again.',
    action: null,
    actionUrl: null,
  };
  
  showError(error);
}

/**
 * Show success state
 */
function showSuccess(memory) {
  const content = document.getElementById('content');
  content.innerHTML = `
    <div class="status success-status">
      <div class="success-icon">✓</div>
      <div class="success-title">Conversation Saved!</div>
      <div class="success-message">
        "${truncate(memory.title, 40)}" has been captured and saved to your memory.
      </div>
    </div>
    <div class="button-group">
      <button class="button button-primary" onclick="window.close()">Close</button>
      <button class="button button-secondary" onclick="window.open('${CONFIG.API_URL}/dashboard', '_blank')">View Dashboard</button>
    </div>
  `;
}

/**
 * Show error state
 */
function showError(error) {
  const content = document.getElementById('content');
  const actionButton = error.action 
    ? `<button class="button button-secondary" onclick="window.open('${error.actionUrl}', '_blank')">${error.action}</button>`
    : '';
    
  content.innerHTML = `
    <div class="status error-status">
      <div class="error-icon">✕</div>
      <div class="error-title">${error.title}</div>
      <div class="error-message">${error.message}</div>
    </div>
    <div class="button-group">
      <button class="button button-primary" onclick="window.location.reload()">Try Again</button>
      ${actionButton}
    </div>
  `;
}

/**
 * Show authentication required state
 */
function showAuthRequired() {
  const content = document.getElementById('content');
  content.innerHTML = `
    <div class="status auth-status">
      <div class="auth-icon">🔒</div>
      <div class="auth-title">Sign In Required</div>
      <div class="auth-message">
        Please sign in to your Zdravo AI account to start capturing conversations.
      </div>
    </div>
    <div class="button-group">
      <button class="button button-primary" onclick="window.open('${CONFIG.API_URL}/auth/login', '_blank')">Sign In</button>
      <button class="button button-secondary" onclick="window.open('${CONFIG.API_URL}/auth/sign-up', '_blank')">Create Account</button>
    </div>
  `;
}

/**
 * Initialize popup
 */
async function initPopup() {
  const auth = await checkAuth();
  
  if (!auth.isAuthenticated) {
    showAuthRequired();
    return;
  }
  
  // Update UI with user info
  const userEmailEl = document.getElementById('userEmail');
  if (userEmailEl && auth.userEmail) {
    userEmailEl.textContent = auth.userEmail;
  }
  
  // Check current platform
  const tab = await getCurrentTab();
  const platform = detectPlatform(tab.url);
  const platformStatusEl = document.getElementById('platformStatus');
  
  if (platformStatusEl) {
    if (platform) {
      platformStatusEl.innerHTML = `<span class="badge badge-success">${getPlatformDisplayName(platform)} Detected</span>`;
    } else {
      platformStatusEl.innerHTML = `<span class="badge badge-warning">Unsupported Site</span>`;
    }
  }
  
  // Set up capture button
  const captureBtn = document.getElementById('captureBtn');
  if (captureBtn) {
    captureBtn.addEventListener('click', captureConversation);
    if (!platform) {
      captureBtn.disabled = true;
      captureBtn.title = 'Navigate to ChatGPT, Claude, Gemini, or Perplexity';
    }
  }
}

/**
 * Utility: Truncate string
 */
function truncate(str, maxLength) {
  if (!str) return '';
  if (str.length <= maxLength) return str;
  return str.substring(0, maxLength) + '...';
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', initPopup);

// Listen for messages from content scripts or background
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'authStateChanged') {
    // Re-check auth when notified by web app
    checkAuth().then(auth => {
      if (!auth.isAuthenticated) {
        showAuthRequired();
      }
    });
  }
  return true;
});
