/**
 * Zdravo AI - Windsurf Integration
 * Extracts conversations from Windsurf IDE
 * 
 * Windsurf is Codeium's AI IDE, similar structure to Cursor
 */

(function() {
  'use strict';

  // Check if we're on Windsurf
  if (!window.location.hostname.includes('windsurf') && 
      !window.location.hostname.includes('codeium')) return;

  // Listen for messages from extension
  window.addEventListener('message', handleExtensionMessage);

  // Ping handler
  window.addEventListener('zdravo-ping', () => {
    window.postMessage({ type: 'zdravo-pong', source: 'windsurf-content' }, '*');
  });

  function handleExtensionMessage(event) {
    if (event.source !== window) return;
    
    const { type, action, data } = event.data || {};
    
    switch (type) {
      case 'ZDRAVO_PING':
        respondToPing();
        break;
      case 'ZDRAVO_GET_CONTENT':
        getWindsurfContent().then(content => {
          window.postMessage({ 
            type: 'ZDRAVO_CONTENT', 
            source: 'windsurf-content',
            data: content 
          }, '*');
        });
        break;
      case 'ZDRAVO_SAVE':
        saveToZdravo(data);
        break;
    }
  }

  function respondToPing() {
    window.postMessage({ 
      type: 'ZDRAVO_PONG', 
      source: 'windsurf-content',
      platform: 'Windsurf',
      version: '1.0.0'
    }, '*');
  }

  async function getWindsurfContent() {
    const conversations = [];
    
    // Windsurf chat interface selectors
    const chatSelectors = [
      '[class*="windsurf-chat"]',
      '[class*="codeium-chat"]',
      '[class*="ai-chat"]',
      '[class*="conversation"]',
      '[class*="chat-message"]',
      '[data-testid*="chat"]',
      '[data-testid*="conversation"]'
    ];
    
    // Extract chat messages
    for (const selector of chatSelectors) {
      const elements = document.querySelectorAll(selector);
      
      elements.forEach((el, index) => {
        const role = el.querySelector('[class*="user"], [class*="assistant"], [class*="ai"], [class*="codeium"]');
        const content = el.textContent || el.innerText;
        
        if (content && content.length > 10) {
          conversations.push({
            id: `windsurf-${Date.now()}-${index}`,
            platform: 'Windsurf',
            role: role?.className?.includes('user') ? 'user' : 'assistant',
            content: cleanContent(content),
            url: window.location.href,
            title: document.title,
            timestamp: new Date().toISOString(),
            metadata: {
              userAgent: navigator.userAgent,
              windowPath: window.location.pathname
            }
          });
        }
      });
    }

    // Extract Flow mode interactions (Windsurf's agentic mode)
    const flowSelectors = [
      '[class*="flow"]',
      '[class*="agent-mode"]',
      '[class*="cascade"]'
    ];

    for (const selector of flowSelectors) {
      const elements = document.querySelectorAll(selector);
      elements.forEach((el, index) => {
        const content = el.textContent?.trim();
        if (content && content.length > 20) {
          conversations.push({
            id: `windsurf-flow-${Date.now()}-${index}`,
            platform: 'Windsurf',
            role: 'assistant',
            content: `[Flow/Cascade] ${content}`,
            url: window.location.href,
            title: document.title,
            timestamp: new Date().toISOString(),
            metadata: { type: 'flow-mode' }
          });
        }
      });
    }

    // Extract Code Editor AI interactions
    const editorSelectors = [
      '[class*="inline-ai"]',
      '[class*="ai-inline"]',
      '[class*="ghost-text"]',
      '[class*="suggestion"]'
    ];

    for (const selector of editorSelectors) {
      const elements = document.querySelectorAll(selector);
      elements.forEach((el, index) => {
        const content = el.textContent?.trim();
        if (content && content.length > 10) {
          conversations.push({
            id: `windsurf-inline-${Date.now()}-${index}`,
            platform: 'Windsurf',
            role: 'assistant',
            content: `[Inline Suggestion] ${content}`,
            url: window.location.href,
            title: document.title,
            timestamp: new Date().toISOString(),
            metadata: { type: 'inline-suggestion' }
          });
        }
      });
    }

    return {
      conversations,
      platform: 'Windsurf',
      url: window.location.href,
      title: document.title,
      timestamp: new Date().toISOString()
    };
  }

  function cleanContent(content) {
    if (!content) return '';
    
    return content
      .replace(/\s+/g, ' ')
      .replace(/[\u0000-\u001F\u007F-\u009F]/g, '')
      .trim();
  }

  function saveToZdravo(data) {
    window.postMessage({
      type: 'ZDRAVO_SAVE_REQUEST',
      source: 'windsurf-content',
      data: {
        ...data,
        platform: 'Windsurf',
        timestamp: new Date().toISOString()
      }
    }, '*');
  }

  // Add Zdravo indicator
  injectWindsurfIndicator();

  function injectWindsurfIndicator() {
    const indicator = document.createElement('div');
    indicator.id = 'zdravo-windsurf-indicator';
    indicator.innerHTML = `
      <style>
        #zdravo-windsurf-indicator {
          position: fixed;
          bottom: 20px;
          right: 20px;
          background: linear-gradient(135deg, #10b981, #3b82f6);
          color: white;
          padding: 8px 16px;
          border-radius: 20px;
          font-family: system-ui, sans-serif;
          font-size: 12px;
          font-weight: 600;
          z-index: 999999;
          cursor: pointer;
          box-shadow: 0 4px 12px rgba(0,0,0,0.15);
          display: flex;
          align-items: center;
          gap: 8px;
          transition: transform 0.2s, box-shadow 0.2s;
        }
        #zdravo-windsurf-indicator:hover {
          transform: translateY(-2px);
          box-shadow: 0 6px 16px rgba(0,0,0,0.2);
        }
        #zdravo-windsurf-indicator .icon {
          width: 16px;
          height: 16px;
        }
      </style>
      <svg class="icon" viewBox="0 0 24 24" fill="currentColor">
        <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/>
      </svg>
      Zdravo AI
    </div>
    `;
    
    indicator.addEventListener('click', () => {
      window.postMessage({ type: 'ZDRAVO_OPEN_POPUP', source: 'windsurf-indicator' }, '*');
    });
    
    document.body.appendChild(indicator);
  }

  console.log('[Zdravo] Windsurf integration loaded');
})();
