/**
 * Zdravo AI - Cursor Integration
 * Extracts conversations from Cursor IDE
 * 
 * Cursor uses a similar structure to ChatGPT but with IDE-specific elements
 */

(function() {
  'use strict';

  // Check if we're on Cursor
  if (!window.location.hostname.includes('cursor.sh')) return;

  // Mascot message system
  const ZDRAVO_STORAGE_KEY = 'zdravo_mascot_state';
  
  // Listen for messages from extension
  window.addEventListener('message', handleExtensionMessage);

  // Ping handler for extension communication
  window.addEventListener('zdravo-ping', () => {
    window.postMessage({ type: 'zdravo-pong', source: 'cursor-content' }, '*');
  });

  function handleExtensionMessage(event) {
    if (event.source !== window) return;
    
    const { type, action, data } = event.data || {};
    
    switch (type) {
      case 'ZDRAVO_PING':
        respondToPing();
        break;
      case 'ZDRAVO_GET_CONTENT':
        getCursorContent().then(content => {
          window.postMessage({ 
            type: 'ZDRAVO_CONTENT', 
            source: 'cursor-content',
            data: content 
          }, '*');
        });
        break;
      case 'ZDRAVO_SAVE':
        saveToZdravo(data);
        break;
    }
  }

  function respondToPing() {
    window.postMessage({ 
      type: 'ZDRAVO_PONG', 
      source: 'cursor-content',
      platform: 'Cursor',
      version: '1.0.0'
    }, '*');
  }

  async function getCursorContent() {
    const conversations = [];
    
    // Try multiple selectors for Cursor's chat interface
    const chatSelectors = [
      '[class*="chat"]',
      '[class*="conversation"]',
      '[class*="message"]',
      '[data-testid*="chat"]',
      '[data-testid*="message"]'
    ];
    
    // Extract chat messages
    for (const selector of chatSelectors) {
      const elements = document.querySelectorAll(selector);
      
      elements.forEach((el, index) => {
        const role = el.querySelector('[class*="user"], [class*="assistant"], [class*="system"]');
        const content = el.textContent || el.innerText;
        
        if (content && content.length > 10) {
          conversations.push({
            id: `cursor-${Date.now()}-${index}`,
            platform: 'Cursor',
            role: role?.className?.includes('user') ? 'user' : 'assistant',
            content: cleanContent(content),
            url: window.location.href,
            title: document.title,
            timestamp: new Date().toISOString(),
            metadata: {
              cursorVersion: navigator.userAgent,
              windowPath: window.location.pathname
            }
          });
        }
      });
    }

    // Also try to get inline edits and agent interactions
    const inlineEditSelectors = [
      '[class*="inline-edit"]',
      '[class*="agent"]',
      '[class*="command"]'
    ];

    for (const selector of inlineEditSelectors) {
      const elements = document.querySelectorAll(selector);
      elements.forEach((el, index) => {
        const content = el.textContent?.trim();
        if (content && content.length > 20) {
          conversations.push({
            id: `cursor-inline-${Date.now()}-${index}`,
            platform: 'Cursor',
            role: 'assistant',
            content: `[Inline Edit/Agent] ${content}`,
            url: window.location.href,
            title: document.title,
            timestamp: new Date().toISOString(),
            metadata: { type: 'inline-edit' }
          });
        }
      });
    }

    // Extract terminal/output if available
    const terminalSelectors = [
      '[class*="terminal"]',
      '[class*="output"]',
      '[class*="console"]'
    ];

    for (const selector of terminalSelectors) {
      const elements = document.querySelectorAll(selector);
      elements.forEach((el, index) => {
        const content = el.textContent?.trim();
        if (content && content.length > 50) {
          conversations.push({
            id: `cursor-terminal-${Date.now()}-${index}`,
            platform: 'Cursor',
            role: 'system',
            content: `[Terminal Output] ${content}`,
            url: window.location.href,
            title: document.title,
            timestamp: new Date().toISOString(),
            metadata: { type: 'terminal' }
          });
        }
      });
    }

    return {
      conversations,
      platform: 'Cursor',
      url: window.location.href,
      title: document.title,
      timestamp: new Date().toISOString()
    };
  }

  function cleanContent(content) {
    if (!content) return '';
    
    return content
      .replace(/\s+/g, ' ')
      .replace(/[\u0000-\u001F\u007F-\u009F]/g, '')
      .trim();
  }

  function saveToZdravo(data) {
    // This will be handled by the extension's background script
    window.postMessage({
      type: 'ZDRAVO_SAVE_REQUEST',
      source: 'cursor-content',
      data: {
        ...data,
        platform: 'Cursor',
        timestamp: new Date().toISOString()
      }
    }, '*');
  }

  // Auto-save functionality (optional - can be enabled/disabled)
  let autoSaveEnabled = localStorage.getItem('zdravo_auto_save') === 'true';
  
  if (autoSaveEnabled) {
    setupAutoSave();
  }

  function setupAutoSave() {
    const observer = new MutationObserver(debounce(() => {
      getCursorContent().then(content => {
        if (content.conversations.length > 0) {
          window.postMessage({
            type: 'ZDRAVO_AUTO_SAVE',
            source: 'cursor-content',
            data: { conversations: content.conversations }
          }, '*');
        }
      });
    }, 5000)); // Save every 5 seconds max

    observer.observe(document.body, {
      childList: true,
      subtree: true,
      characterData: true
    });
  }

  function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
      const later = () => {
        clearTimeout(timeout);
        func(...args);
      };
      clearTimeout(timeout);
      timeout = setTimeout(later, wait);
    };
  }

  // Add Zdravo indicator in the page
  injectCursorIndicator();

  function injectCursorIndicator() {
    const indicator = document.createElement('div');
    indicator.id = 'zdravo-cursor-indicator';
    indicator.innerHTML = `
      <style>
        #zdravo-cursor-indicator {
          position: fixed;
          bottom: 20px;
          right: 20px;
          background: linear-gradient(135deg, #fbbf24, #3b82f6);
          color: white;
          padding: 8px 16px;
          border-radius: 20px;
          font-family: system-ui, sans-serif;
          font-size: 12px;
          font-weight: 600;
          z-index: 999999;
          cursor: pointer;
          box-shadow: 0 4px 12px rgba(0,0,0,0.15);
          display: flex;
          align-items: center;
          gap: 8px;
          transition: transform 0.2s, box-shadow 0.2s;
        }
        #zdravo-cursor-indicator:hover {
          transform: translateY(-2px);
          box-shadow: 0 6px 16px rgba(0,0,0,0.2);
        }
        #zdravo-cursor-indicator .icon {
          width: 16px;
          height: 16px;
        }
      </style>
      <svg class="icon" viewBox="0 0 24 24" fill="currentColor">
        <path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/>
      </svg>
      Zdravo AI
    </div>
    `;
    
    indicator.addEventListener('click', () => {
      window.postMessage({ type: 'ZDRAVO_OPEN_POPUP', source: 'cursor-indicator' }, '*');
    });
    
    document.body.appendChild(indicator);
  }

  console.log('[Zdravo] Cursor integration loaded');
})();
