/**
 * Zdravo AI - Chrome Extension Background Service Worker
 * Handles context menu, token management, and memory saving
 */

const API_URL = 'https://www.zdravoai.com';

// Generate and persist a device ID for guest mode
function getDeviceId() {
  return new Promise((resolve) => {
    chrome.storage.local.get(['zdravo_device_id'], (result) => {
      let deviceId = result.zdravo_device_id;
      if (deviceId) {
        resolve(deviceId);
      } else {
        deviceId = 'ext_' + Date.now() + Math.random().toString(36).substr(2, 9);
        chrome.storage.local.set({ zdravo_device_id: deviceId }, () => {
          resolve(deviceId);
        });
      }
    });
  });
}

// Get auth token from storage
function getAuthToken() {
  return new Promise((resolve) => {
    chrome.storage.local.get(['zdravo_auth_token'], (result) => {
      resolve(result.zdravo_auth_token || null);
    });
  });
}

// Store auth token
function setAuthToken(token) {
  return new Promise((resolve) => {
    chrome.storage.local.set({ zdravo_auth_token: token }, resolve);
  });
}

// Clear auth token
function clearAuthToken() {
  return new Promise((resolve) => {
    chrome.storage.local.remove(['zdravo_auth_token'], resolve);
  });
}

// Show notification
function showNotification(title, message, isError = false) {
  chrome.notifications.create({
    type: 'basic',
    iconUrl: 'icons/icon128.png',
    title: title,
    message: message,
    priority: 2
  });
}

// Setup context menu
chrome.runtime.onInstalled.addListener(() => {
  chrome.contextMenus.create({
    id: "zdravo-save-selection",
    title: "Add Memory to Zdravo",
    contexts: ["selection"]
  });
});

// Handle context menu clicks
chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  if (info.menuItemId !== "zdravo-save-selection" || !info.selectionText) {
    return;
  }

  const title = `Note from: ${tab.title}`;
  const content = info.selectionText;
  const url = tab.url;

  console.log('ZDRAVO EXT: Attempting to save selection.');

  try {
    const authToken = await getAuthToken();
    
    if (authToken) {
      console.log('ZDRAVO EXT: Auth token found. Saving as user.');
      await saveAsUser(title, content, url, authToken);
    } else {
      console.log('ZDRAVO EXT: No auth token. Saving as guest.');
      await saveAsGuest(title, content, url);
    }
  } catch (error) {
    console.error('ZDRAVO EXT: Main save error:', error);
    showNotification('Zdravo AI Error', 'Could not save memory. Please try again.', true);
  }
});

// Save as authenticated user
async function saveAsUser(title, content, url, authToken) {
  try {
    const response = await fetch(`${API_URL}/api/memories`, {
      method: 'POST',
      headers: { 
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${authToken}`
      },
      body: JSON.stringify({
        title: title,
        content: content,
        url: url,
        sourcePlatform: 'web',
        tags: ['note', 'extension']
      })
    });

    const data = await response.json();

    if (response.ok) {
      showNotification('Zdravo AI', 'Memory saved successfully!');
    } else {
      // Token might be expired
      if (response.status === 401) {
        await clearAuthToken();
        showNotification('Zdravo AI', 'Session expired. Please log in again.', true);
      } else {
        throw new Error(data.error || `API Error: ${response.status}`);
      }
    }
  } catch (error) {
    console.error('ZDRAVO EXT: User save failed:', error);
    const message = error.message.includes('limit')
      ? 'Free user limit reached. Upgrade for unlimited saves.'
      : 'Could not save memory. Are you logged in to Zdravo?';
    showNotification('Zdravo AI Error', message, true);
  }
}

// Save as guest
async function saveAsGuest(title, content, url) {
  const deviceId = await getDeviceId();
  try {
    const response = await fetch(`${API_URL}/api/guest/memories`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'X-Device-ID': deviceId
      },
      body: JSON.stringify({
        title: title,
        content: content,
        url: url,
        sourcePlatform: 'web',
        deviceId: deviceId
      })
    });

    const data = await response.json();

    if (response.ok) {
      const remaining = data.remaining || '?';
      showNotification('Zdravo AI', `Saved as temporary guest memory (${remaining} left).`);
    } else {
      throw new Error(data.error || `API Error: ${response.status}`);
    }
  } catch (error) {
    console.error('ZDRAVO EXT: Guest save failed:', error);
    const message = error.message.includes('limit')
      ? 'Guest save limit reached. Sign up for free to continue.'
      : 'Could not save guest memory.';
    showNotification('Zdravo AI Error', message, true);
  }
}

// Listen for messages from popup
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'getAuthToken') {
    getAuthToken().then(token => sendResponse({ token }));
    return true; // Keep channel open for async response
  }
  
  if (request.action === 'setAuthToken') {
    setAuthToken(request.token).then(() => sendResponse({ success: true }));
    return true;
  }
  
  if (request.action === 'clearAuthToken') {
    clearAuthToken().then(() => sendResponse({ success: true }));
    return true;
  }
});
